<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Visitor;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class BadgeController extends Controller
{
    /**
     * Generate visitor badge (returns HTML/PDF data)
     */
    public function generate(string $visitorId): JsonResponse
    {
        $visitor = Visitor::with('host')->findOrFail($visitorId);

        // Generate badge data
        $badgeData = [
            'visitorName' => $visitor->full_name,
            'companyName' => $visitor->company_name,
            'purpose' => $visitor->purpose,
            'hostName' => $visitor->host->name,
            'hostDepartment' => $visitor->host->department,
            'checkInTime' => $visitor->time_in->format('M j, Y g:i A'),
            'badgeId' => 'V-' . str_pad($visitor->id, 6, '0', STR_PAD_LEFT),
            'qrCode' => $this->generateQRCode($visitor->id),
        ];

        return response()->json([
            'success' => true,
            'data' => $badgeData,
        ]);
    }

    /**
     * Generate QR code data (base64 encoded)
     */
    private function generateQRCode(int $visitorId): string
    {
        // Simple QR code generation using data URI
        // In production, use a proper QR code library like SimpleSoftwareIO/simple-qrcode
        $data = json_encode([
            'visitor_id' => $visitorId,
            'timestamp' => now()->toISOString(),
        ]);

        // For now, return a placeholder. In production, generate actual QR code
        // Using a simple approach - you can integrate a QR code library
        return 'data:image/svg+xml;base64,' . base64_encode($this->generateQRCodeSVG($data));
    }

    /**
     * Generate simple QR code SVG (placeholder - use proper library in production)
     */
    private function generateQRCodeSVG(string $data): string
    {
        // This is a placeholder. In production, use a QR code library
        // For now, return a simple SVG pattern
        $hash = md5($data);
        $size = 200;
        
        $svg = '<svg width="' . $size . '" height="' . $size . '" xmlns="http://www.w3.org/2000/svg">';
        $svg .= '<rect width="' . $size . '" height="' . $size . '" fill="white"/>';
        
        // Generate a simple pattern based on hash
        $cellSize = 10;
        for ($i = 0; $i < strlen($hash); $i++) {
            $x = ($i % ($size / $cellSize)) * $cellSize;
            $y = (int)($i / ($size / $cellSize)) * $cellSize;
            if (hexdec($hash[$i]) > 7) {
                $svg .= '<rect x="' . $x . '" y="' . $y . '" width="' . $cellSize . '" height="' . $cellSize . '" fill="black"/>';
            }
        }
        
        $svg .= '</svg>';
        return $svg;
    }

    /**
     * Print badge (returns printable HTML)
     */
    public function print(string $visitorId): \Illuminate\Http\Response
    {
        $visitor = Visitor::with('host')->findOrFail($visitorId);

        $badgeHtml = view('badges.visitor', [
            'visitor' => $visitor,
            'badgeId' => 'V-' . str_pad($visitor->id, 6, '0', STR_PAD_LEFT),
            'checkInTime' => $visitor->time_in->format('M j, Y g:i A'),
        ])->render();

        return response($badgeHtml)->header('Content-Type', 'text/html');
    }

    /**
     * Get badge template
     */
    public function template(): JsonResponse
    {
        return response()->json([
            'success' => true,
            'data' => [
                'template' => 'default',
                'fields' => [
                    'visitorName',
                    'companyName',
                    'purpose',
                    'hostName',
                    'checkInTime',
                    'badgeId',
                    'qrCode',
                ],
            ],
        ]);
    }
}
