<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Visitor;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class RealtimeController extends Controller
{
    /**
     * Server-Sent Events endpoint for real-time updates
     */
    public function stream(Request $request)
    {
        // Check authentication via token in query or header
        $token = $request->query('token') ?? $request->bearerToken();
        if ($token) {
            try {
                auth()->setToken($token);
            } catch (\Exception $e) {
                // Invalid token, but continue for public events
            }
        }

        // Set headers for SSE
        header('Content-Type: text/event-stream');
        header('Cache-Control: no-cache');
        header('Connection: keep-alive');
        header('X-Accel-Buffering: no'); // Disable nginx buffering
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Credentials: true');

        // Close output buffering
        if (ob_get_level()) {
            ob_end_clean();
        }

        $lastEventId = (int) $request->header('Last-Event-ID', 0);
        $userId = auth()->id() ?? 0;

        // Send initial connection message
        echo "data: " . json_encode([
            'type' => 'connected',
            'message' => 'Connected to real-time updates',
            'timestamp' => now()->toISOString(),
        ]) . "\n\n";
        flush();

        // Keep connection alive and send updates
        $startTime = time();
        $maxDuration = 300; // 5 minutes max connection time

        while (time() - $startTime < $maxDuration) {
            // Check for new events (both user-specific and global)
            $userEvents = Cache::get("realtime_events_{$userId}", []);
            $globalEvents = Cache::get('realtime_events_global', []);
            $allEvents = array_merge($userEvents, $globalEvents);
            ksort($allEvents); // Sort by event ID

            foreach ($allEvents as $eventId => $event) {
                if ($eventId > $lastEventId) {
                    echo "id: {$eventId}\n";
                    echo "data: " . json_encode($event) . "\n\n";
                    flush();
                    $lastEventId = $eventId;
                }
            }

            // Clean processed events
            if (!empty($userEvents)) {
                $newUserEvents = array_filter($userEvents, function ($id) use ($lastEventId) {
                    return $id > $lastEventId;
                }, ARRAY_FILTER_USE_KEY);
                Cache::put("realtime_events_{$userId}", $newUserEvents, 60);
            }

            // Send heartbeat every 30 seconds
            if ((time() - $startTime) % 30 === 0) {
                echo ": heartbeat\n\n";
                flush();
            }

            sleep(2); // Check every 2 seconds
        }

        // Send disconnect message
        echo "data: " . json_encode([
            'type' => 'disconnected',
            'message' => 'Connection closed',
        ]) . "\n\n";
        flush();
    }

    /**
     * Broadcast an event to all connected clients
     */
    public static function broadcast(string $type, array $data, ?int $userId = null): void
    {
        $event = [
            'type' => $type,
            'data' => $data,
            'timestamp' => now()->toISOString(),
        ];

        if ($userId) {
            // Send to specific user
            $events = Cache::get("realtime_events_{$userId}", []);
            $events[time()] = $event;
            Cache::put("realtime_events_{$userId}", $events, 60);
        } else {
            // Broadcast to all users (get all active user IDs from cache or database)
            // For simplicity, we'll store a global event
            $events = Cache::get('realtime_events_global', []);
            $events[time()] = $event;
            Cache::put('realtime_events_global', $events, 60);
        }
    }
}
