<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Visitor;
use App\Models\VisitorNote;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class VisitorNoteController extends Controller
{
    /**
     * Get all notes for a visitor.
     */
    public function index(string $visitorId): JsonResponse
    {
        $visitor = Visitor::findOrFail($visitorId);
        
        $notes = VisitorNote::where('visitor_id', $visitorId)
            ->with('user:id,name,email')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $notes->map(function ($note) {
                return [
                    'id' => $note->id,
                    'note' => $note->note,
                    'isPrivate' => $note->is_private,
                    'createdAt' => $note->created_at->toISOString(),
                    'user' => $note->user ? [
                        'id' => $note->user->id,
                        'name' => $note->user->name,
                        'email' => $note->user->email,
                    ] : null,
                ];
            }),
        ]);
    }

    /**
     * Store a new note for a visitor.
     */
    public function store(Request $request, string $visitorId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'note' => 'required|string|max:1000',
            'is_private' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $visitor = Visitor::findOrFail($visitorId);

        $note = VisitorNote::create([
            'visitor_id' => $visitorId,
            'user_id' => Auth::id(),
            'note' => $request->note,
            'is_private' => $request->get('is_private', true),
        ]);

        $note->load('user:id,name,email');

        return response()->json([
            'success' => true,
            'message' => 'Note added successfully',
            'data' => [
                'id' => $note->id,
                'note' => $note->note,
                'isPrivate' => $note->is_private,
                'createdAt' => $note->created_at->toISOString(),
                'user' => $note->user ? [
                    'id' => $note->user->id,
                    'name' => $note->user->name,
                    'email' => $note->user->email,
                ] : null,
            ],
        ], 201);
    }

    /**
     * Update a note.
     */
    public function update(Request $request, string $visitorId, string $noteId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'note' => 'required|string|max:1000',
            'is_private' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $note = VisitorNote::where('visitor_id', $visitorId)
            ->where('id', $noteId)
            ->firstOrFail();

        // Only allow the creator or admin to update
        if ($note->user_id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized to update this note',
            ], 403);
        }

        $note->update([
            'note' => $request->note,
            'is_private' => $request->get('is_private', $note->is_private),
        ]);

        $note->load('user:id,name,email');

        return response()->json([
            'success' => true,
            'message' => 'Note updated successfully',
            'data' => [
                'id' => $note->id,
                'note' => $note->note,
                'isPrivate' => $note->is_private,
                'createdAt' => $note->created_at->toISOString(),
                'user' => $note->user ? [
                    'id' => $note->user->id,
                    'name' => $note->user->name,
                    'email' => $note->user->email,
                ] : null,
            ],
        ]);
    }

    /**
     * Delete a note.
     */
    public function destroy(string $visitorId, string $noteId): JsonResponse
    {
        $note = VisitorNote::where('visitor_id', $visitorId)
            ->where('id', $noteId)
            ->firstOrFail();

        // Only allow the creator or admin to delete
        if ($note->user_id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized to delete this note',
            ], 403);
        }

        $note->delete();

        return response()->json([
            'success' => true,
            'message' => 'Note deleted successfully',
        ]);
    }
}
