<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ScheduledVisit extends Model
{
    protected $fillable = [
        'visitor_name',
        'visitor_email',
        'visitor_phone',
        'company_name',
        'purpose',
        'host_id',
        'scheduled_date',
        'scheduled_time',
        'status',
        'check_in_id',
        'notes',
        'confirmation_token',
    ];

    protected $casts = [
        'scheduled_date' => 'date',
        'scheduled_time' => 'datetime:H:i',
    ];

    /**
     * Get the host (staff member) for this scheduled visit
     */
    public function host(): BelongsTo
    {
        return $this->belongsTo(Staff::class, 'host_id');
    }

    /**
     * Get the actual check-in record if visitor has checked in
     */
    public function checkIn(): BelongsTo
    {
        return $this->belongsTo(Visitor::class, 'check_in_id');
    }

    /**
     * Check if visit is in the past
     */
    public function isPast(): bool
    {
        $visitDateTime = $this->scheduled_date->setTimeFromTimeString($this->scheduled_time);
        return $visitDateTime->isPast();
    }

    /**
     * Check if visit is today
     */
    public function isToday(): bool
    {
        return $this->scheduled_date->isToday();
    }

    /**
     * Check if visit is upcoming
     */
    public function isUpcoming(): bool
    {
        return !$this->isPast() && $this->status === 'scheduled';
    }
}
