<?php

namespace App\Services;

use App\Models\AuditLog;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AuditLogService
{
    /**
     * Log an action
     */
    public static function log(
        string $action,
        ?Model $model = null,
        ?array $oldValues = null,
        ?array $newValues = null,
        ?string $description = null,
        ?Request $request = null
    ): AuditLog {
        $request = $request ?? request();
        $user = Auth::user();

        return AuditLog::create([
            'user_id' => $user?->id,
            'action' => $action,
            'model_type' => $model ? get_class($model) : null,
            'model_id' => $model?->id,
            'description' => $description ?? self::generateDescription($action, $model),
            'old_values' => $oldValues,
            'new_values' => $newValues,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'url' => $request->fullUrl(),
            'method' => $request->method(),
        ]);
    }

    /**
     * Generate a description from action and model
     */
    private static function generateDescription(string $action, ?Model $model): string
    {
        if (!$model) {
            return ucfirst($action);
        }

        $modelName = class_basename($model);
        return ucfirst($action) . ' ' . $modelName;
    }

    /**
     * Log authentication events
     */
    public static function logAuth(string $action, ?string $email = null, ?Request $request = null): AuditLog
    {
        $request = $request ?? request();
        $user = Auth::user();

        return AuditLog::create([
            'user_id' => $user?->id,
            'action' => $action,
            'description' => $action === 'login' 
                ? "User logged in: {$email}" 
                : ($action === 'logout' ? "User logged out" : ucfirst($action)),
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'url' => $request->fullUrl(),
            'method' => $request->method(),
        ]);
    }
}
