<?php

namespace Database\Seeders;

use App\Models\Permission;
use App\Models\Role;
use Illuminate\Database\Seeder;

class RolePermissionSeeder extends Seeder
{
    public function run(): void
    {
        // Create Permissions
        $permissions = [
            // Visitor permissions
            ['name' => 'visitors.view', 'display_name' => 'View Visitors', 'description' => 'Can view visitor list and details'],
            ['name' => 'visitors.create', 'display_name' => 'Create Visitors', 'description' => 'Can check in new visitors'],
            ['name' => 'visitors.update', 'display_name' => 'Update Visitors', 'description' => 'Can update visitor information'],
            ['name' => 'visitors.delete', 'display_name' => 'Delete Visitors', 'description' => 'Can delete visitor records'],
            ['name' => 'visitors.checkout', 'display_name' => 'Checkout Visitors', 'description' => 'Can check out visitors'],
            ['name' => 'visitors.export', 'display_name' => 'Export Visitors', 'description' => 'Can export visitor data'],
            
            // Staff permissions
            ['name' => 'staff.view', 'display_name' => 'View Staff', 'description' => 'Can view staff list'],
            ['name' => 'staff.create', 'display_name' => 'Create Staff', 'description' => 'Can create new staff members'],
            ['name' => 'staff.update', 'display_name' => 'Update Staff', 'description' => 'Can update staff information'],
            ['name' => 'staff.delete', 'display_name' => 'Delete Staff', 'description' => 'Can delete staff members'],
            
            // Analytics permissions
            ['name' => 'analytics.view', 'display_name' => 'View Analytics', 'description' => 'Can view analytics and reports'],
            
            // Settings permissions
            ['name' => 'settings.view', 'display_name' => 'View Settings', 'description' => 'Can view system settings'],
            ['name' => 'settings.update', 'display_name' => 'Update Settings', 'description' => 'Can update system settings'],
            
            // User management permissions
            ['name' => 'users.view', 'display_name' => 'View Users', 'description' => 'Can view user list'],
            ['name' => 'users.create', 'display_name' => 'Create Users', 'description' => 'Can create new users'],
            ['name' => 'users.update', 'display_name' => 'Update Users', 'description' => 'Can update user information'],
            ['name' => 'users.delete', 'display_name' => 'Delete Users', 'description' => 'Can delete users'],
            
            // Audit log permissions
            ['name' => 'audit.view', 'display_name' => 'View Audit Logs', 'description' => 'Can view audit logs'],
            ['name' => 'audit.export', 'display_name' => 'Export Audit Logs', 'description' => 'Can export audit logs'],
        ];

        foreach ($permissions as $permission) {
            Permission::firstOrCreate(
                ['name' => $permission['name']],
                $permission
            );
        }

        // Create Roles
        $superAdmin = Role::firstOrCreate(
            ['name' => 'super_admin'],
            [
                'display_name' => 'Super Administrator',
                'description' => 'Full system access with all permissions',
            ]
        );

        $admin = Role::firstOrCreate(
            ['name' => 'admin'],
            [
                'display_name' => 'Administrator',
                'description' => 'Can manage visitors, staff, and view analytics',
            ]
        );

        $receptionist = Role::firstOrCreate(
            ['name' => 'receptionist'],
            [
                'display_name' => 'Receptionist',
                'description' => 'Can check in/out visitors and view basic information',
            ]
        );

        $viewer = Role::firstOrCreate(
            ['name' => 'viewer'],
            [
                'display_name' => 'Viewer',
                'description' => 'Read-only access to visitors and analytics',
            ]
        );

        // Assign all permissions to super admin
        $superAdmin->permissions()->sync(Permission::pluck('id'));

        // Assign permissions to admin (all except user management)
        $admin->permissions()->sync(
            Permission::whereNotIn('name', ['users.view', 'users.create', 'users.update', 'users.delete'])->pluck('id')
        );

        // Assign permissions to receptionist
        $receptionist->permissions()->sync(
            Permission::whereIn('name', [
                'visitors.view',
                'visitors.create',
                'visitors.checkout',
                'staff.view',
            ])->pluck('id')
        );

        // Assign permissions to viewer
        $viewer->permissions()->sync(
            Permission::whereIn('name', [
                'visitors.view',
                'staff.view',
                'analytics.view',
            ])->pluck('id')
        );

        $this->command->info('Roles and permissions created successfully!');
    }
}
